/*

	Copyright (C) 2006  Dave Keck

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

*/

#include "aw_lib.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdbool.h>

/* Comment-out this next line to use the callback scenario. */

//#define BUFFER_EVENTS

#define allOff		(unsigned char *)"0000000000000000"
#define oneOn		(unsigned char *)"1000000000000000"
#define allOn		(unsigned char *)"1111111111111111"

static unsigned char decimalArray[2], binaryArray[8];

void boardConnected(AW_EVENT event);
void boardDisconnected(AW_EVENT event);
void boardDataChanged(AW_EVENT event);

static void bail(char *reason, char n)
{

	printf(reason);
	
	/* We're feelin courteous so we'll say our graceful goodbyes
	   by resetting the board's pins before storming out. */
	
	aw_setDirectionsOfPinSets(0, 0, 0);
	
	aw_close();
	
	exit(n);

}

static void signalHandler()
{

	bail("Received SIG_INT, resetting pins & exiting...\n", 0);

}

static void prepareBoard(unsigned char boardNumber)
{

	aw_binaryToDecimal(allOn, 2, decimalArray);
	
	if (aw_setDirectionsOfPinSets(boardNumber, decimalArray[0], decimalArray[1]))
		printf("Unable to set the IO pins' directions. Continuing...\n");
	
	else
		printf("Set IO pins' directions.\n");

}

int main()
{

	/* eventList is only used in the BUFFER_EVENTS scenerio. */
	AW_EVENT_LIST eventList;
	unsigned char numberOfBoards = 0;
	bool b = false;
	
	signal(SIGINT, signalHandler);
	
	/* Initialize the ActiveWire driver. */
	
	if (aw_init())
		bail("Unable to initialize driver. Exiting...\n", 1);
	
	else
		printf("Initialized driver.\n");
	
	/* Retrieve number of connected ActiveWire boards. */
	
	if (aw_numberOfConnectedBoards(&numberOfBoards))
		bail("Unable to retrieve number of boards connected. Exiting...\n", 2);
	
	if (!numberOfBoards)
		printf("No ActiveWire boards currently connected. Writing anyway...\n");
	
	/* Setup chosen event-retrieval method. (We can either buffer events
	   and access them when we feel like it, or receive callbacks that
	   notify us when an event occurs. */
	
	#ifdef BUFFER_EVENTS
	
		printf("### Buffer Events Scenerio ###\n");
		
		if (aw_setEventDeliveryMethod(AW_EVENT_DELIV_BUFFER,
									  NULL,
									  NULL,
									  NULL,
									  0))
	
	#else
	
		printf("### Callback Events Scenerio ###\n");
		
		if (aw_setEventDeliveryMethod(AW_EVENT_DELIV_CALLBACK,
									  boardConnected,
									  boardDisconnected,
									  boardDataChanged,
									  0))
	
	#endif
			bail("Unable to set event delivery method. Exiting...\n", 3);
	
	else
		printf("Set callbacks.\n");
	
	printf("Number of connected boards: %d.\n", numberOfBoards);
	
	/* Prepare our ActiveWire board. This simply resets the pins' directions. */
	
	prepareBoard(0);
	
	printf("Toggling IO pin 0...\n");
	
	#ifdef BUFFER_EVENTS
	
		eventList = aw_getEvents();
	
	#endif
	
	/* Begin our loop which will toggle the first pins' value. */
	
	while (1)
	{
	
		if (!b)
			aw_binaryToDecimal(oneOn, 2, decimalArray);
		
		else
			aw_binaryToDecimal(allOff, 2, decimalArray);
		
		b = !b;
		
		if (aw_writeData(0, decimalArray, 2))
			printf("Unable to write data. Exiting...\n");
		
		else
			printf("Wrote data...\n");
		
		#ifdef BUFFER_EVENTS
		
			/* Give a little time for the driver to register the event. */
			
			usleep(10000);
			
			if (eventList->numberOfEvents)
			{
			
				/* This loads the binary representation of the board's data when the
				   event occurred into our binaryArray, so we can access each bit
				   independently. */
				
				aw_decimalToBinary(eventList->events[eventList->numberOfEvents - 1]->boardData, 1, binaryArray);
				
				printf("Last event:\n	Board: %d\n	Type: %d\n	Time: (s: %d, ns: %d)\n	IO Pin 0: %d\n", eventList->events[eventList->numberOfEvents - 1]->boardNumber,
																										 eventList->events[eventList->numberOfEvents - 1]->eventType,
																										 (int)eventList->events[eventList->numberOfEvents - 1]->timestamp.tv_sec,
																										 (int)eventList->events[eventList->numberOfEvents - 1]->timestamp.tv_nsec,
																										 binaryArray[0]);
				
				aw_clearEvents();
			
			}
		
		#endif
		
		/* Wait one second between each time we toggle the pin. */
		
		sleep(1);
		
	}
	
	return 0;

}

/* Callbacks */

void boardConnected(AW_EVENT event)
{

	printf("Board %d connected.\n", event->boardNumber);
	
	prepareBoard(event->boardNumber);

}

void boardDisconnected(AW_EVENT event)
{

	printf("Board %d disconnected.\n", event->boardNumber);

}

void boardDataChanged(AW_EVENT event)
{

	aw_decimalToBinary(event->boardData, 1, binaryArray);
	
	printf("Board data changed! IO pin 0 data: %d.\n", binaryArray[0]);

}